<?php
session_start();
require_once '../config/db.php';
require_once '../config/config.php';
require_once '../auth/auth.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set JSON header
header('Content-Type: application/json');

// Check if user has permission to view customers
if (!hasPermission('view_customers')) {
    http_response_code(403);
    echo json_encode(['error' => 'Permission denied']);
    exit;
}

if (!isset($_GET['customer_id']) || !isset($_GET['sale_amount'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Customer ID and sale amount are required']);
    exit;
}

$customerId = intval($_GET['customer_id']);
$saleAmount = floatval($_GET['sale_amount']);

try {
    // Get customer credit information using the EXACT same logic as customers.php
    $stmt = $pdo->prepare("
        SELECT c.id, c.name, c.credit_limit, c.status,
               COALESCE(credit_summary.total_credit_used, 0) as total_credit_used,
               COALESCE(credit_summary.total_paid_and_waived, 0) as total_paid_and_waived,
               COALESCE(credit_summary.total_balance, 0) as current_usage
        FROM customers c
        LEFT JOIN (
            SELECT 
                ct.customer_id,
                SUM(ct.amount) as total_credit_used,
                SUM(COALESCE(paid_amounts.paid, 0) + COALESCE(waived_amounts.waived, 0)) as total_paid_and_waived,
                SUM(ct.amount - COALESCE(paid_amounts.paid, 0) - COALESCE(waived_amounts.waived, 0)) as total_balance
            FROM credit_transactions ct
            LEFT JOIN (
                SELECT 
                    cp.credit_transaction_id,
                    SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END) as paid
                FROM credit_payments cp
                GROUP BY cp.credit_transaction_id
            ) paid_amounts ON ct.id = paid_amounts.credit_transaction_id
            LEFT JOIN (
                SELECT 
                    cp.credit_transaction_id,
                    SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END) as waived
                FROM credit_payments cp
                GROUP BY cp.credit_transaction_id
            ) waived_amounts ON ct.id = waived_amounts.credit_transaction_id
            WHERE ct.type = 'credit'
            AND (ct.amount - COALESCE(paid_amounts.paid, 0) - COALESCE(waived_amounts.waived, 0)) > 0.001
            GROUP BY ct.customer_id
        ) credit_summary ON c.id = credit_summary.customer_id
        WHERE c.id = ?
    ");
    
    $stmt->execute([$customerId]);
    $customer = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$customer) {
        http_response_code(404);
        echo json_encode(['error' => 'Customer not found']);
        exit;
    }
    
    // Check if customer is suspended
    if ($customer['status'] === 'suspended') {
        echo json_encode([
            'success' => false,
            'error' => 'Customer is suspended and cannot make credit purchases',
            'customer_name' => $customer['name'],
            'credit_limit' => floatval($customer['credit_limit']),
            'available_credit' => 0,
            'current_usage' => 0
        ]);
        exit;
    }
    
    // Calculate available credit using the EXACT same logic as customers.php
    $creditLimit = floatval($customer['credit_limit']);
    $currentCreditUsage = floatval($customer['current_usage'] ?? 0); // This now matches customers.php "Credit Amount Used" column
    $availableCredit = $creditLimit - $currentCreditUsage;
    
    // Check if customer has credit enabled
    if ($creditLimit <= 0) {
        echo json_encode([
            'success' => false,
            'error' => 'Customer does not have credit enabled',
            'customer_name' => $customer['name'],
            'credit_limit' => $creditLimit,
            'available_credit' => 0,
            'current_usage' => $currentCreditUsage
        ]);
        exit;
    }
    
    // Check if sale amount exceeds available credit
    if ($saleAmount > $availableCredit) {
        echo json_encode([
            'success' => false,
            'error' => 'Sale amount exceeds available credit',
            'customer_name' => $customer['name'],
            'credit_limit' => $creditLimit,
            'available_credit' => $availableCredit,
            'current_usage' => $currentCreditUsage,
            'sale_amount' => $saleAmount,
            'deficit' => $saleAmount - $availableCredit
        ]);
        exit;
    }
    
    // Calculate usage percentage
    $usageAfterSale = $currentCreditUsage + $saleAmount;
    $usagePercent = ($usageAfterSale / $creditLimit) * 100;
    
    // Determine warning level
    $warningLevel = 'none';
    if ($usagePercent >= 90) {
        $warningLevel = 'critical';
    } elseif ($usagePercent >= 75) {
        $warningLevel = 'warning';
    } elseif ($usagePercent >= 50) {
        $warningLevel = 'moderate';
    }
    
    // Credit check passed
    echo json_encode([
        'success' => true,
        'customer_name' => $customer['name'],
        'credit_limit' => $creditLimit,
        'available_credit' => $availableCredit,
        'current_usage' => $currentCreditUsage,
        'sale_amount' => $saleAmount,
        'usage_after_sale' => $usageAfterSale,
        'usage_percent_after_sale' => $usagePercent,
        'available_after_sale' => $availableCredit - $saleAmount,
        'warning_level' => $warningLevel
    ]);
    
} catch (PDOException $e) {
    error_log("Credit check database error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Database error occurred']);
} catch (Exception $e) {
    error_log("Credit check general error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'An unexpected error occurred']);
} 